/*
 * WiFi NTP Clock for Arduino UNO R4 WiFi with LED Matrix
 *
 * このプログラムはWiFi経由でNTPサーバーから時刻を取得し、
 * ボード上のLEDマトリックスとシリアルモニターに現在時刻を表示します。
 *
 * 使用方法:
 * 1. SSID と PASSWORD を自分のWiFi情報に変更してください
 * 2. タイムゾーン(GMT_OFFSET_SEC)を必要に応じて調整してください
 *    日本標準時(JST)の場合: 32400 (9時間 = 9 * 3600秒)
 */

#include <WiFiS3.h>
#include <NTPClient.h>
#include <WiFiUdp.h>
#include "Arduino_LED_Matrix.h"

// WiFi設定 - ここを変更してください
const char* SSID = "YOUR_WIFI_SSID";      // WiFiのSSID
const char* PASSWORD = "YOUR_WIFI_PASSWORD";  // WiFiのパスワード

// NTP設定
const long GMT_OFFSET_SEC = 32400;  // 日本標準時(JST) = GMT+9
const int DAYLIGHT_OFFSET_SEC = 0;   // サマータイム(日本は不要)

WiFiUDP ntpUDP;
NTPClient timeClient(ntpUDP, "pool.ntp.org", GMT_OFFSET_SEC, 60000);
ArduinoLEDMatrix matrix;

// スクロール用の変数
int scrollOffset = -12;  // 画面幅分マイナスにして右から開始
unsigned long lastScrollTime = 0;
const int SCROLL_DELAY = 150;  // スクロール速度（ミリ秒）

// 数字のフォント定義 (3x5ドット)
const uint8_t digits[10][5] = {
  {0b111, 0b101, 0b101, 0b101, 0b111}, // 0
  {0b010, 0b110, 0b010, 0b010, 0b111}, // 1
  {0b111, 0b001, 0b111, 0b100, 0b111}, // 2
  {0b111, 0b001, 0b111, 0b001, 0b111}, // 3
  {0b101, 0b101, 0b111, 0b001, 0b001}, // 4
  {0b111, 0b100, 0b111, 0b001, 0b111}, // 5
  {0b111, 0b100, 0b111, 0b101, 0b111}, // 6
  {0b111, 0b001, 0b001, 0b001, 0b001}, // 7
  {0b111, 0b101, 0b111, 0b101, 0b111}, // 8
  {0b111, 0b101, 0b111, 0b001, 0b111}  // 9
};

// LEDマトリックスに時刻をスクロール表示する関数
void displayTimeOnMatrix(int hours, int minutes) {
  // 仮想的な大きなフレームバッファ（18列分: HH:MM = 3+1+3+1+3+1+3 + スペース）
  uint8_t virtualFrame[8][20] = {0};

  // 時の10の位
  int h1 = hours / 10;
  // 時の1の位
  int h2 = hours % 10;
  // 分の10の位
  int m1 = minutes / 10;
  // 分の1の位
  int m2 = minutes % 10;

  int xPos = 0;

  // 時の10の位を描画
  for (int y = 0; y < 5; y++) {
    for (int x = 0; x < 3; x++) {
      if (digits[h1][y] & (1 << (2 - x))) {
        virtualFrame[y + 1][xPos + x] = 1;
      }
    }
  }
  xPos += 4;  // 3ドット + 1スペース

  // 時の1の位を描画
  for (int y = 0; y < 5; y++) {
    for (int x = 0; x < 3; x++) {
      if (digits[h2][y] & (1 << (2 - x))) {
        virtualFrame[y + 1][xPos + x] = 1;
      }
    }
  }
  xPos += 3;

  // コロン (:) を描画
  virtualFrame[2][xPos] = 1;
  virtualFrame[4][xPos] = 1;
  xPos += 2;  // コロン1ドット + 1スペース

  // 分の10の位を描画
  for (int y = 0; y < 5; y++) {
    for (int x = 0; x < 3; x++) {
      if (digits[m1][y] & (1 << (2 - x))) {
        virtualFrame[y + 1][xPos + x] = 1;
      }
    }
  }
  xPos += 4;  // 3ドット + 1スペース

  // 分の1の位を描画
  for (int y = 0; y < 5; y++) {
    for (int x = 0; x < 3; x++) {
      if (digits[m2][y] & (1 << (2 - x))) {
        virtualFrame[y + 1][xPos + x] = 1;
      }
    }
  }
  xPos += 3;

  // 表示用のフレームバッファ（12x8）
  uint8_t frame[8][12] = {0};

  // スクロールオフセットを適用して実際のフレームにコピー
  for (int y = 0; y < 8; y++) {
    for (int x = 0; x < 12; x++) {
      int srcX = x + scrollOffset;
      if (srcX >= 0 && srcX < 20) {
        frame[y][x] = virtualFrame[y][srcX];
      }
    }
  }

  matrix.renderBitmap(frame, 8, 12);

  // スクロール処理
  unsigned long currentTime = millis();
  if (currentTime - lastScrollTime > SCROLL_DELAY) {
    scrollOffset++;
    // 全体の幅（xPos）が画面外に消えたら右端からリセット
    if (scrollOffset > xPos) {
      scrollOffset = -12;  // 画面幅分マイナスにして右から再開始
    }
    lastScrollTime = currentTime;
  }
}

void setup() {
  Serial.begin(115200);
  while (!Serial) {
    ; // シリアルポート接続待ち
  }

  // LEDマトリックス初期化
  matrix.begin();

  Serial.println("\n=== Arduino UNO R4 WiFi Clock ===");
  Serial.println("WiFi接続中...");

  // WiFi接続
  WiFi.begin(SSID, PASSWORD);

  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 30) {
    delay(500);
    Serial.print(".");
    attempts++;
  }

  if (WiFi.status() == WL_CONNECTED) {
    Serial.println("\nWiFi接続成功!");
    Serial.print("IPアドレス: ");
    Serial.println(WiFi.localIP());

    // NTPクライアント開始
    timeClient.begin();
    Serial.println("NTPサーバーから時刻を取得中...");
    timeClient.update();
    Serial.println("時刻取得完了!\n");
  } else {
    Serial.println("\nWiFi接続失敗!");
    Serial.println("SSID と PASSWORD を確認してください。");
  }
}

void loop() {
  if (WiFi.status() == WL_CONNECTED) {
    // NTPサーバーから時刻を更新
    timeClient.update();

    // 現在時刻を取得
    unsigned long epochTime = timeClient.getEpochTime();

    // 時刻を分解
    int hours = timeClient.getHours();
    int minutes = timeClient.getMinutes();
    int seconds = timeClient.getSeconds();

    // 日付を計算（gmtimeの代わりに手動計算）
    // 1970年1月1日からの経過日数を計算
    unsigned long days = epochTime / 86400;  // 86400秒 = 1日
    int weekDay = (days + 4) % 7;  // 1970/1/1は木曜日(4)

    // 年を計算
    int year = 1970;
    unsigned long daysInYear;
    while (true) {
      // うるう年判定
      bool isLeapYear = (year % 4 == 0 && year % 100 != 0) || (year % 400 == 0);
      daysInYear = isLeapYear ? 366 : 365;

      if (days >= daysInYear) {
        days -= daysInYear;
        year++;
      } else {
        break;
      }
    }

    // 月を計算
    int daysInMonth[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
    bool isLeapYear = (year % 4 == 0 && year % 100 != 0) || (year % 400 == 0);
    if (isLeapYear) {
      daysInMonth[1] = 29;  // うるう年の2月
    }

    int month = 1;
    for (int i = 0; i < 12; i++) {
      if (days >= daysInMonth[i]) {
        days -= daysInMonth[i];
        month++;
      } else {
        break;
      }
    }

    int day = days + 1;  // 日は1から始まる

    // 曜日の配列
    const char* weekDays[] = {"日", "月", "火", "水", "木", "金", "土"};

    // LEDマトリックスに時刻をスクロール表示
    displayTimeOnMatrix(hours, minutes);

    // シリアルモニターに表示（1秒ごと）
    static unsigned long lastSerialUpdate = 0;
    if (millis() - lastSerialUpdate >= 1000) {
      Serial.print("\r");  // カーソルを行頭に戻す
      Serial.print(year);
      Serial.print("年");
      Serial.print(month < 10 ? "0" : "");
      Serial.print(month);
      Serial.print("月");
      Serial.print(day < 10 ? "0" : "");
      Serial.print(day);
      Serial.print("日(");
      Serial.print(weekDays[weekDay]);
      Serial.print(") ");

      Serial.print(hours < 10 ? "0" : "");
      Serial.print(hours);
      Serial.print(":");
      Serial.print(minutes < 10 ? "0" : "");
      Serial.print(minutes);
      Serial.print(":");
      Serial.print(seconds < 10 ? "0" : "");
      Serial.print(seconds);
      Serial.print("   ");  // 前の表示をクリア

      lastSerialUpdate = millis();
    }

    delay(10);  // スクロールをスムーズにするため短い遅延
  } else {
    Serial.println("WiFi接続が切断されました。再接続中...");
    WiFi.begin(SSID, PASSWORD);
    delay(5000);
  }
}
